//
//  MPAdViewDelegate.h
//
//  Copyright 2018-2021 Twitter, Inc.
//  Licensed under the MoPub SDK License Agreement
//  http://www.mopub.com/legal/sdk-license-agreement/
//

#import <SDK_Lib/MPMoPubAd.h>

@class MPAdView;

/**
 The delegate of an `MPAdView` object must adopt the `MPAdViewDelegate` protocol. It must
 implement `viewControllerForPresentingModalView` to provide a root view controller from which
 the ad view should present modal content.

 Optional methods of this protocol allow the delegate to be notified of banner success or
 failure, as well as other lifecycle events.
 */

@protocol MPAdViewDelegate <MPMoPubAdDelegate>

@required

/** @name Managing Modal Content Presentation */

/**
 Asks the delegate for a view controller to use for presenting modal content, such as the in-app
 browser that can appear when an ad is tapped.

 @return A view controller that should be used for presenting modal content.
 */
- (UIViewController *)viewControllerForPresentingModalView;

@optional

/** @name Detecting When a Banner Ad is Loaded */

/**
 Sent when an ad view successfully loads an ad.

 Your implementation of this method should insert the ad view into the view hierarchy, if you
 have not already done so.

 @param view The ad view sending the message.
 */
- (void)adViewDidLoadAd:(MPAdView *)view __attribute__((deprecated("Deprecated; please use adViewDidLoadAd:adSize: instead.")));

/**
 Sent when an ad view successfully loads an ad.

 Your implementation of this method should insert the ad view into the view hierarchy, if you
 have not already done so.

 @param view The ad view sending the message.
 @param adSize The size of the ad that was successfully loaded. It is recommended to resize
 the @c MPAdView frame to match the height of the loaded ad.
 */
- (void)adViewDidLoadAd:(MPAdView *)view adSize:(CGSize)adSize;

/**
 Sent when an ad view fails to load an ad.

 To avoid displaying blank ads, you should hide the ad view in response to this message.

 @param view The ad view sending the message.
 */
- (void)adViewDidFailToLoadAd:(MPAdView *)view __attribute__((deprecated("Deprecated; please use adView:didFailToLoadAdWithError: instead.")));

/**
 Sent when an ad view fails to load an ad.

 To avoid displaying blank ads, you should hide the ad view in response to this message.

 @param view The ad view sending the message.
 @param error The error
 */
- (void)adView:(MPAdView *)view didFailToLoadAdWithError:(NSError *)error;

/** @name Detecting When a User Interacts With the Ad View */

/**
 Sent when an ad view is about to present modal content.

 This method is called when the user taps on the ad view. Your implementation of this method
 should pause any application activity that requires user interaction.

 @param view The ad view sending the message.
 @see `didDismissModalViewForAd:`
 */
- (void)willPresentModalViewForAd:(MPAdView *)view;

/**
 Sent when an ad view has dismissed its modal content, returning control to your application.

 Your implementation of this method should resume any application activity that was paused
 in response to `willPresentModalViewForAd:`.

 @param view The ad view sending the message.
 @see `willPresentModalViewForAd:`
 */
- (void)didDismissModalViewForAd:(MPAdView *)view;

/**
 Sent when a user is about to leave your application as a result of tapping
 on an ad.

 Your application will be moved to the background shortly after this method is called.

 @param view The ad view sending the message.
 */
- (void)willLeaveApplicationFromAd:(MPAdView *)view;

@end
